<?php
// api.php
declare(strict_types=1);
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type');
require_once __DIR__ . '/db.php';

$action = $_GET['action'] ?? ($_POST['action'] ?? '');

function json_response($data, int $code=200): void {
    http_response_code($code);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

if ($action === 'settings' && $_SERVER['REQUEST_METHOD'] === 'GET') {
    json_response(get_settings());
}

if ($action === 'settings' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $payload = json_decode(file_get_contents('php://input'), true) ?? $_POST;
    json_response(update_settings($payload));
}

if ($action === 'employees') {
    $stmt = db()->query('SELECT * FROM employees ORDER BY name');
    json_response($stmt->fetchAll(PDO::FETCH_ASSOC));
}

if ($action === 'attendance') {
    $date = $_GET['date'] ?? (new DateTime())->format('Y-m-d');
    json_response(build_daily_report($date));
}

if ($action === 'summary') {
    $person_id = $_GET['person_id'] ?? '';
    if (!$person_id) json_response(['error' => 'person_id is required'], 400);
    $from = $_GET['from'] ?? (new DateTime('first day of this month'))->format('Y-m-d');
    $to = $_GET['to'] ?? (new DateTime())->format('Y-m-d');
    $settings = get_settings();
    $days = [];
    $cur = new DateTime($from);
    $end = new DateTime($to);
    $end->modify('+1 day');

    $totalFine = 0; $totalMinutes = 0; $totalBonus = 0;
    while ($cur < $end) {
        $dateStr = $cur->format('Y-m-d');
        $rows = build_daily_report($dateStr);
        foreach ($rows as $r) {
            if ($r['person_id'] === $person_id) {
                $row = ['date' => $dateStr] + $r;
                $days[] = $row;
                $totalFine += (int)$r['fine'];
                $totalMinutes += (int)$r['duration_min'];
                if ((int)$r['duration_min'] >= 10*60) {
                    $totalBonus += (float)($settings['bonus_per_day'] ?? 0);
                }
            }
        }
        $cur->modify('+1 day');
    }
    json_response([
        'person_id' => $person_id,
        'from' => $from,
        'to' => $to,
        'total_days' => count($days),
        'total_minutes' => $totalMinutes,
        'total_fine' => (int)round($totalFine),
        'total_bonus' => (int)round($totalBonus),
        'days' => $days
    ]);
}

if ($action === 'event' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $payload = json_decode(file_get_contents('php://input'), true) ?? $_POST;
    $person_id = trim((string)($payload['person_id'] ?? ''));
    $name = (string)($payload['name'] ?? '');
    $department = (string)($payload['department'] ?? '');
    $ts = (string)($payload['ts'] ?? '');
    $status = (string)($payload['status'] ?? '');
    $checkpoint = (string)($payload['checkpoint'] ?? '');
    if (!$person_id || !$ts) json_response(['error' => 'person_id and ts required'], 400);
    upsert_employee($person_id, $name, $department);
    insert_event([
        'person_id' => $person_id,
        'name' => $name,
        'department' => $department,
        'ts' => $ts,
        'status' => $status,
        'checkpoint' => $checkpoint
    ]);
    json_response(['ok' => true]);
}

if ($action === 'import' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!isset($_FILES['file'])) json_response(['error' => 'CSV file is required as "file"'], 400);
    $tmp = $_FILES['file']['tmp_name'];
    $raw = file_get_contents($tmp);
    if ($raw === false) json_response(['error' => 'Failed to read upload'], 500);

    // Detect encoding and delimiter
    $encoding = mb_detect_encoding($raw, ['UTF-8','UTF-16LE','UTF-16BE','Windows-1251','Windows-1252'], true);
    if ($encoding && $encoding !== 'UTF-8') {
        $raw = iconv($encoding, 'UTF-8//IGNORE', $raw);
    }
    $firstLine = strtok($raw, "\n");
    $delim = (strpos($firstLine, "\t") !== false) ? "\t" : ",";

    $lines = preg_split("/\r\n|\n|\r/", $raw);
    $headers = [];
    $imported = 0; $employees = 0; $errors = 0;

    foreach ($lines as $i => $line) {
        if ($line === '') continue;
        $row = str_getcsv($line, $delim);
        if ($i === 0) { $headers = $row; continue; }
        $assoc = [];
        foreach ($headers as $k => $h) {
            $assoc[$h] = $row[$k] ?? '';
        }
        try {
            $person_id = preg_replace("/^'+/", "", (string)($assoc['Person ID'] ?? $assoc['PersonID'] ?? $assoc['ID'] ?? ''));
            $name = (string)($assoc['Name'] ?? $assoc['Full Name'] ?? '');
            $department = (string)($assoc['Department'] ?? '');
            $time = (string)($assoc['Time'] ?? $assoc['Timestamp'] ?? '');
            $status = (string)($assoc['Attendance Status'] ?? $assoc['Status'] ?? $assoc['Custom Name'] ?? '');
            if ($status === '' && isset($assoc['Custom Name'])) $status = (string)$assoc['Custom Name'];
            $checkpoint = (string)($assoc['Attendance Check Point'] ?? $assoc['Device'] ?? '');

            if (!$person_id || !$time) continue;

            // Normalize datetime
            $ts = (new DateTime($time))->format('Y-m-d H:i:s');

            upsert_employee($person_id, $name, $department); $employees++;
            insert_event([
                'person_id' => $person_id,
                'name' => $name,
                'department' => $department,
                'ts' => $ts,
                'status' => $status,
                'checkpoint' => $checkpoint
            ]);
            $imported++;
        } catch (\Throwable $e) {
            $errors++;
        }
    }
    json_response(['imported' => $imported, 'employees' => $employees, 'errors' => $errors]);
}

// default: not found
json_response(['error' => 'Unknown action'], 404);
