<?php
// db.php
declare(strict_types=1);

function db(): PDO {
    static $pdo = null;
    if ($pdo === null) {
        $pdo = new PDO('sqlite:' . __DIR__ . '/attendance.db');
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $pdo->exec('PRAGMA journal_mode = WAL');
        $pdo->exec('
        CREATE TABLE IF NOT EXISTS employees(
            person_id TEXT PRIMARY KEY,
            name TEXT,
            department TEXT
        );
        CREATE TABLE IF NOT EXISTS events(
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            person_id TEXT,
            name TEXT,
            department TEXT,
            ts TEXT,
            status TEXT,
            checkpoint TEXT
        );
        CREATE TABLE IF NOT EXISTS settings(
            id INTEGER PRIMARY KEY CHECK (id=1),
            workday_start TEXT DEFAULT "09:00",
            grace_minutes INTEGER DEFAULT 10,
            fine_per_minute REAL DEFAULT 300,
            fine_per_hour REAL DEFAULT 10000,
            fine_per_day REAL DEFAULT 100000,
            bonus_per_minute REAL DEFAULT 0,
            bonus_per_hour REAL DEFAULT 0,
            bonus_per_day REAL DEFAULT 0
        );
        ');
        $stmt = $pdo->prepare('INSERT OR IGNORE INTO settings(id) VALUES (1)');
        $stmt->execute();
    }
    return $pdo;
}

function get_settings(): array {
    $stmt = db()->query('SELECT * FROM settings WHERE id=1');
    return $stmt->fetch(PDO::FETCH_ASSOC) ?: [];
}

function update_settings(array $s): array {
    $current = get_settings();
    $data = array_merge($current, $s);
    $stmt = db()->prepare('UPDATE settings SET
        workday_start=:workday_start,
        grace_minutes=:grace_minutes,
        fine_per_minute=:fine_per_minute,
        fine_per_hour=:fine_per_hour,
        fine_per_day=:fine_per_day,
        bonus_per_minute=:bonus_per_minute,
        bonus_per_hour=:bonus_per_hour,
        bonus_per_day=:bonus_per_day
        WHERE id=1');
    $stmt->execute([
        ':workday_start' => $data['workday_start'] ?? '09:00',
        ':grace_minutes' => (int)($data['grace_minutes'] ?? 10),
        ':fine_per_minute' => (float)($data['fine_per_minute'] ?? 0),
        ':fine_per_hour' => (float)($data['fine_per_hour'] ?? 0),
        ':fine_per_day' => (float)($data['fine_per_day'] ?? 0),
        ':bonus_per_minute' => (float)($data['bonus_per_minute'] ?? 0),
        ':bonus_per_hour' => (float)($data['bonus_per_hour'] ?? 0),
        ':bonus_per_day' => (float)($data['bonus_per_day'] ?? 0),
    ]);
    return get_settings();
}

function upsert_employee(string $person_id, string $name, string $department): void {
    $stmt = db()->prepare('INSERT INTO employees(person_id, name, department) VALUES(?,?,?)
        ON CONFLICT(person_id) DO UPDATE SET name=excluded.name, department=excluded.department');
    $stmt->execute([$person_id, $name, $department]);
}

function insert_event(array $ev): void {
    $stmt = db()->prepare('INSERT INTO events(person_id, name, department, ts, status, checkpoint)
        VALUES(?,?,?,?,?,?)');
    $stmt->execute([
        $ev['person_id'] ?? '',
        $ev['name'] ?? '',
        $ev['department'] ?? '',
        $ev['ts'] ?? '',
        $ev['status'] ?? '',
        $ev['checkpoint'] ?? ''
    ]);
}

function events_range(string $fromISO, string $toISO, ?string $person_id=null): array {
    if ($person_id) {
        $stmt = db()->prepare('SELECT * FROM events WHERE ts >= ? AND ts < ? AND person_id=? ORDER BY ts ASC');
        $stmt->execute([$fromISO, $toISO, $person_id]);
    } else {
        $stmt = db()->prepare('SELECT * FROM events WHERE ts >= ? AND ts < ? ORDER BY ts ASC');
        $stmt->execute([$fromISO, $toISO]);
    }
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

function build_daily_report(string $date): array {
    $start = new DateTime($date . ' 00:00:00');
    $end = clone $start; $end->modify('+1 day');
    $settings = get_settings();
    $events = events_range($start->format('Y-m-d H:i:s'), $end->format('Y-m-d H:i:s'));

    // group by person
    $byPerson = [];
    foreach ($events as $e) {
        $byPerson[$e['person_id']][] = $e;
    }

    $rows = [];
    foreach ($byPerson as $pid => $list) {
        // sort just in case
        usort($list, fn($a,$b) => strcmp($a['ts'],$b['ts']));
        $firstIn = null;
        $lastOut = null;
        foreach ($list as $e) {
            if (!$firstIn && preg_match('/in/i', $e['status'] ?? '')) $firstIn = $e;
            if (preg_match('/out/i', $e['status'] ?? '')) $lastOut = $e;
        }
        if (!$firstIn) $firstIn = $list[0] ?? null;
        if (!$lastOut) $lastOut = end($list) ?: null;

        $checkIn = $firstIn ? new DateTime($firstIn['ts']) : null;
        $checkOut = $lastOut ? new DateTime($lastOut['ts']) : null;
        $durationMin = ($checkIn && $checkOut) ? max(0, (int) round(($checkOut->getTimestamp() - $checkIn->getTimestamp()) / 60)) : 0;

        $targetStart = new DateTime($date . ' ' . ($settings['workday_start'] ?? '09:00'));
        $latenessMin = $checkIn ? max(0, (int) round(($checkIn->getTimestamp() - $targetStart->getTimestamp()) / 60)) : 0;
        $lateAfterGrace = max(0, $latenessMin - (int)($settings['grace_minutes'] ?? 0));

        $fineMin = ((float)($settings['fine_per_minute'] ?? 0)) * $lateAfterGrace;
        $fineHour = ((float)($settings['fine_per_hour'] ?? 0)) * ($lateAfterGrace / 60.0);
        $fineDay = $lateAfterGrace >= 8*60 ? ((float)($settings['fine_per_day'] ?? 0)) : 0;
        $fineTotal = (int) round($fineMin + $fineHour + $fineDay);

        $rows[] = [
            'person_id' => $pid,
            'name' => $list[0]['name'] ?? '',
            'department' => $list[0]['department'] ?? '',
            'check_in' => $checkIn ? $checkIn->format('H:i:s') : '-',
            'check_out' => $checkOut ? $checkOut->format('H:i:s') : '-',
            'duration_min' => $durationMin,
            'lateness_min' => $latenessMin,
            'fine' => $fineTotal
        ];
    }
    return $rows;
}
