<?php
// public/categories.php
require_once __DIR__ . '/auth_stub.php';
require_once __DIR__ . '/db.php';

session_start();
$userName = current_user_name() ?: 'Admin';
?>
<!DOCTYPE html>
<html lang="uz">
<head>
  <meta charset="UTF-8">
  <title>Kategoriyalar – Markirovka tizimi</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">

  <!-- Bootstrap 5 -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <!-- FontAwesome -->
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css" rel="stylesheet">
  <!-- Toastr -->
  <link href="https://cdnjs.cloudflare.com/ajax/libs/toastr.js/latest/toastr.min.css" rel="stylesheet">

  <style>
    :root {
      --brand-main: #0d47a1;
      --brand-soft: #e5edff;
      --brand-accent: #22c55e;
      --bg-soft: #f3f4f8;
      --card-bg: rgba(255,255,255,0.96);
      --border-soft: #d1d9e6;
      --text-main: #0b1220;
      --text-muted: #6b7280;
    }

    body {
      min-height: 100vh;
      margin: 0;
      background: radial-gradient(circle at top left, #c7d2fe 0, transparent 45%),
                  radial-gradient(circle at bottom right, #bfdbfe 0, transparent 45%),
                  #eef2ff;
      color: var(--text-main);
    }

    .page-wrapper {
      max-width: 1200px;
      margin: 0 auto;
    }

    .navbar {
      background: rgba(13,71,161,0.96);
      backdrop-filter: blur(10px);
      box-shadow: 0 10px 25px rgba(15,23,42,0.35);
    }
    .navbar-brand, .navbar-nav .nav-link, .navbar-text {
      color: #fff !important;
    }
    .navbar-nav .nav-link {
      font-size: 0.92rem;
      opacity: 0.9;
    }
    .navbar-nav .nav-link.active {
      opacity: 1;
      font-weight: 600;
      border-bottom: 2px solid #fff;
    }

    .user-chip {
      display: inline-flex;
      align-items: center;
      gap: 0.5rem;
      padding: 0.15rem 0.75rem;
      background: rgba(15,23,42,0.2);
      border-radius: 999px;
      color: #e5edff;
      font-size: 0.85rem;
    }
    .user-avatar {
      width: 26px;
      height: 26px;
      border-radius: 999px;
      background: linear-gradient(135deg,#22c55e,#0ea5e9);
      display: inline-flex;
      align-items: center;
      justify-content: center;
      font-size: 0.9rem;
      color: #fff;
      font-weight: 700;
    }
    .user-role-badge {
      padding: 0.05rem 0.5rem;
      border-radius: 999px;
      background: rgba(15,23,42,0.35);
      font-size: 0.7rem;
      text-transform: uppercase;
      letter-spacing: 0.05em;
    }

    .card {
      border-radius: 18px;
      box-shadow: 0 16px 40px rgba(15,23,42,0.08);
      border: 1px solid rgba(148,163,184,0.35);
      background: var(--card-bg);
    }

    .form-label {
      font-weight: 600;
      color: var(--text-main);
    }

    .pill-heading {
      font-size: 0.8rem;
      text-transform: uppercase;
      letter-spacing: 0.1em;
      color: var(--text-muted);
    }

    .badge-status {
      font-size: 0.75rem;
      border-radius: 999px;
      padding: 0.2rem 0.55rem;
    }

    .table thead {
      background: #eff3ff;
      font-size: 0.88rem;
    }
    .table tbody tr {
      font-size: 0.86rem;
      transition: background 0.12s ease;
    }
    .table tbody tr:hover {
      background: #f9fafb;
    }

    .search-input {
      max-width: 260px;
    }

    .cat-img-thumb {
      width: 46px;
      height: 46px;
      object-fit: cover;
      border-radius: 10px;
      border: 1px solid #e5e7eb;
    }

    /* Global loader */
    #global-loader {
      position: fixed;
      inset: 0;
      background: rgba(15,23,42,0.25);
      display: none;
      align-items: center;
      justify-content: center;
      z-index: 2000;
    }
    #global-loader .spinner-border {
      width: 3rem;
      height: 3rem;
    }
    #global-loader .loader-text {
      color: #e5edff;
      margin-top: 0.75rem;
      font-size: 0.9rem;
    }

    @media (max-width: 767.98px) {
      .page-wrapper {
        padding-inline: 0.4rem;
      }
      .search-input {
        max-width: 100%;
      }
    }
  </style>
</head>
<body>

<?php
  $active = 'categories';
  require __DIR__ . '/navbar.php';
?>

<!-- Global loader -->
<div id="global-loader" class="d-flex flex-column align-items-center justify-content-center">
  <div class="spinner-border text-light" role="status">
    <span class="visually-hidden">Yuklanmoqda...</span>
  </div>
  <div class="loader-text">Ma’lumotlar yuklanmoqda...</div>
</div>

<div class="page-wrapper">
  <div class="py-4">

    <div class="d-flex flex-column flex-md-row justify-content-between align-items-md-center mb-3">
      <div>
        <h3 class="mb-1">
          Kategoriyalar boshqaruvi
         
        </h3>
        <div class="text-muted">
          Mahsulot guruhlarini yaratish va rasm bilan bog‘lash (markirovka uchun).
        </div>
      </div>
      <div class="mt-3 mt-md-0">
        <button type="button" class="btn btn-primary" id="btn-open-create">
          <i class="fa-solid fa-plus"></i> Yangi kategoriya
        </button>
      </div>
    </div>

    <!-- Form + List card (export area) -->
    <div class="card mb-4" id="export-area">
      <div class="card-body">

        <!-- Form header -->
        <div class="d-flex justify-content-between align-items-center mb-3">
          <div>
            <div class="pill-heading mb-1">Kiritish</div>
            <h5 class="mb-0" id="form-title-text">Yangi kategoriya qo‘shish</h5>
          </div>
          <button id="btn-new" type="button" class="btn btn-outline-secondary btn-sm" style="display:none;">
            <i class="fa-solid fa-plus" data-permission="categories.create"></i> Yangi qo‘shish
          </button>
        </div>

        <!-- Inline filter / search / export -->
        <div class="row g-3 mb-3 align-items-end">
          <div class="col-md-4">
            <label class="form-label small text-muted" for="filter_status">Holat bo‘yicha filtrlash</label>
            <select id="filter_status" class="form-select form-select-sm">
              <option value="">Holat (hammasi)</option>
              <option value="1">Faqat faol</option>
              <option value="0">Faqat no-faol</option>
            </select>
          </div>
          <div class="col-md-4">
            <label class="form-label small text-muted" for="search">Qidirish</label>
            <input type="text" id="search" class="form-control form-control-sm search-input"
                   placeholder="Qidirish (kategoriya nomi)...">
          </div>
          <div class="col-md-4 d-flex justify-content-md-end">
            <div class="btn-group">
              <button type="button" class="btn btn-outline-secondary btn-sm dropdown-toggle" data-bs-toggle="dropdown">
                <i class="fa-solid fa-file-arrow-down"></i> Export
              </button>
              <ul class="dropdown-menu dropdown-menu-end">
                <li><button class="dropdown-item" type="button" onclick="exportData('csv')">
                  <i class="fa-solid fa-file-csv me-2"></i>CSV
                </button></li>
                <li><button class="dropdown-item" type="button" onclick="exportData('xls')">
                  <i class="fa-regular fa-file-excel me-2"></i>XLS
                </button></li>
                <li><button class="dropdown-item" type="button" onclick="exportData('xlsx')">
                  <i class="fa-solid fa-file-excel me-2"></i>XLSX
                </button></li>
                <li><button class="dropdown-item" type="button" onclick="exportData('pdf')">
                  <i class="fa-regular fa-file-pdf me-2"></i>PDF
                </button></li>
                <li><button class="dropdown-item" type="button" onclick="exportData('doc')">
                  <i class="fa-regular fa-file-word me-2"></i>DOC
                </button></li>
                <li><hr class="dropdown-divider"></li>
                <li><button class="dropdown-item" type="button" onclick="exportData('png')">
                  <i class="fa-regular fa-image me-2"></i>PNG
                </button></li>
                <li><button class="dropdown-item" type="button" onclick="exportData('jpg')">
                  <i class="fa-regular fa-image me-2"></i>JPG
                </button></li>
              </ul>
            </div>
          </div>
        </div>

        <!-- List table -->
        <div class="table-responsive">
          <table class="table table-hover align-middle" id="categories-table">
            <thead>
            <tr>
              <th>ID</th>
              <th>Kategoriya</th>
              <th>Rasm</th>
              <th>Holat</th>
              <th>Yaratilgan</th>
              <th class="text-end">Amallar</th>
            </tr>
            </thead>
            <tbody>
              <tr><td colspan="6" class="text-center py-3 text-muted">Yuklanmoqda...</td></tr>
            </tbody>
          </table>
        </div>

        <!-- Pagination info -->
        <div class="d-flex justify-content-between align-items-center mt-3 flex-wrap gap-2">
          <div class="text-muted small" id="pagination-info"></div>
          <nav>
            <ul class="pagination pagination-sm mb-0" id="pagination"></ul>
          </nav>
        </div>

      </div>
    </div>

  </div>
</div>

<!-- CATEGORY MODAL (Create/Edit) -->
<div class="modal fade" id="categoryModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-lg modal-dialog-centered">
    <form id="category-form" class="modal-content" enctype="multipart/form-data">
      <div class="modal-header">
        <h5 class="modal-title" id="categoryModalLabel">Yangi kategoriya qo‘shish</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body">
        <input type="hidden" id="cat_id" name="id" value="">

        <div class="row g-3">
          <div class="col-md-6">
            <label class="form-label" for="cat_name">Kategoriya nomi *</label>
            <input type="text" class="form-control" id="cat_name" name="name" required>
          </div>

          <div class="col-md-6">
            <label class="form-label" for="cat_image_file">Rasm *</label>
            <input
              type="file"
              class="form-control"
              id="cat_image_file"
              name="image"
              accept="image/*"
            >
            <input type="hidden" id="existing_image" name="existing_image" value="">

            <div class="mt-2">
              <img
                id="cat_image_preview"
                src=""
                alt="Kategoriya rasmi"
                class="img-thumbnail d-none"
                style="max-height: 140px;"
              >
              <div
                id="cat_image_placeholder"
                class="border rounded text-muted small text-center py-3"
              >
                Rasm tanlanmagan
              </div>
            </div>
            <div class="form-text">
              Yangi kategoriyada rasm majburiy. Tahrirlashda rasmni o‘zgartirish ixtiyoriy.
            </div>
          </div>

          <div class="col-md-6">
            <label class="form-label d-block">Holat</label>
            <div class="form-check form-switch">
              <input class="form-check-input" type="checkbox" role="switch" id="cat_is_active" name="is_active" checked>
              <label class="form-check-label" for="cat_is_active">Faol (sistemada ko‘rinadi)</label>
            </div>
          </div>
        </div>

      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Bekor qilish</button>
        <button type="submit" class="btn btn-primary" id="btn-save">
          <i class="fa-solid fa-floppy-disk"></i> Saqlash
        </button>
      </div>
    </form>
  </div>
</div>

<!-- DELETE CONFIRM MODAL -->
<div class="modal fade" id="deleteModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <div class="modal-content">
      <div class="modal-header bg-danger text-white">
        <h5 class="modal-title"><i class="fa-solid fa-triangle-exclamation"></i> Kategoriyani o‘chirish</h5>
        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body">
        <p class="mb-2">Mazkur kategoriyani rostdan ham o‘chirmoqchimisiz?</p>
        <p class="fw-semibold" id="deleteCategoryName"></p>
        <input type="hidden" id="deleteCategoryId" value="">
        <div class="alert alert-warning mb-0 small">
          <i class="fa-solid fa-circle-info"></i>
          Eslatma: agar bu kategoriya mahsulotlar bilan bog‘langan bo‘lsa,
          backendda cheklov qo‘yish tavsiya etiladi (foreign key).
        </div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Bekor qilish</button>
        <button type="button" class="btn btn-danger" id="btn-confirm-delete">
          <i class="fa-solid fa-trash-can"></i> O‘chirish
        </button>
      </div>
    </div>
  </div>
</div>

<script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/toastr.js/latest/toastr.min.js"></script>

<!-- jsPDF + autoTable + html2canvas (PDF / PNG / JPG uchun) -->
<script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf-autotable/3.8.1/jspdf.plugin.autotable.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js"></script>

<script>
  const apiUrl = '/mark2/api/categories.php'; // kerak bo‘lsa yo‘lini o‘zgartirasiz

  let allCategories = [];
  let filteredCategories = [];
  let currentPage = 1;
  const perPage = 10;

  let categoryModal;
  let deleteModal;

  toastr.options = {
    positionClass: "toast-top-right",
    timeOut: 3000
  };

  // Loader helperlari
  function showLoader() {
    const el = document.getElementById('global-loader');
    if (!el) return;
    el.style.setProperty('display', 'flex', 'important');
    el.style.opacity = '1';
  }

  function hideLoader() {
    const el = document.getElementById('global-loader');
    if (!el) return;
    el.style.setProperty('display', 'none', 'important');
    el.style.opacity = '0';
  }

  // Rasm preview helperlari
  function updateImagePreviewFromUrl(url) {
  const $img = $('#cat_image_preview');
  const $ph  = $('#cat_image_placeholder');

  let src = (url || '').trim();

  if (src) {
    // Agar absolute URL bo'lsa (http/https) — o'zgartirmaymiz
    if (!/^https?:\/\//i.test(src)) {
      // Agar allaqachon /mark2/ bilan boshlanmagan bo'lsa
      if (!src.startsWith('/mark2/')) {
        // Boshlanishida / bo'lmasa, qo'shib qo'yamiz
        if (!src.startsWith('/')) {
          src = '/' + src;
        }
        // Oldiga /mark2 ni qo'shamiz
        src = '/mark2' + src;
      }
    }

    $img.attr('src', src).removeClass('d-none');
    $ph.addClass('d-none');
  } else {
    $img.attr('src', '').addClass('d-none');
    $ph.removeClass('d-none');
  }
}


  function updateImagePreviewFromFile(input) {
    const file = input.files && input.files[0];
    const $img = $('#cat_image_preview');
    const $ph  = $('#cat_image_placeholder');

    if (!file) {
      return;
    }
    const reader = new FileReader();
    reader.onload = function (e) {
      $img.attr('src', e.target.result).removeClass('d-none');
      $ph.addClass('d-none');
    };
    reader.readAsDataURL(file);
  }

  $(function () {
    categoryModal = new bootstrap.Modal(document.getElementById('categoryModal'));
    deleteModal   = new bootstrap.Modal(document.getElementById('deleteModal'));

    // Fayl tanlanganda preview
    $('#cat_image_file').on('change', function () {
      updateImagePreviewFromFile(this);
    });

    loadCategories();

    $('#btn-open-create').on('click', function () {
      resetForm();
      $('#categoryModalLabel').text('Yangi kategoriya qo‘shish');
      categoryModal.show();
    });

    $('#category-form').on('submit', function (e) {
      e.preventDefault();
      saveCategory();
    });

    $('#btn-confirm-delete').on('click', function () {
      confirmDelete();
    });

    $('#btn-new').on('click', function () {
      resetForm();
    });

    $('#filter_status').on('change', function () {
      safeApplyFilters();
    });

    $('#search').on('input', function () {
      safeApplyFilters();
    });
  });

  // ==== Kategoriyalarni yuklash ====
  function loadCategories() {
    $('#categories-table tbody').html(
      '<tr><td colspan="6" class="text-center py-3 text-muted">Yuklanmoqda...</td></tr>'
    );

    showLoader();

    $.ajax({
      url: apiUrl,
      method: 'GET',
      dataType: 'json',
      data: { action: 'list' }
    })
      .done(function (res) {
        if (!res || typeof res !== 'object') {
          toastr.error('Noto‘g‘ri server javobi (JSON emas)');
          $('#categories-table tbody').html(
            '<tr><td colspan="6" class="text-center py-3 text-danger">Serverdan noto‘g‘ri javob keldi</td></tr>'
          );
          return;
        }
        if (res.status !== 'ok') {
          toastr.error(res.message || 'Xatolik');
          $('#categories-table tbody').html(
            '<tr><td colspan="6" class="text-center py-3 text-danger">Ma’lumot yuklab bo‘lmadi</td></tr>'
          );
          return;
        }
        if (!Array.isArray(res.categories)) {
          toastr.error('Serverdan noto‘g‘ri formatdagi "categories" keldi');
          $('#categories-table tbody').html(
            '<tr><td colspan="6" class="text-center py-3 text-danger">Ma’lumot formati xato</td></tr>'
          );
          return;
        }

        allCategories = res.categories;
        safeApplyFilters();
      })
      .fail(function () {
        toastr.error('Kategoriyalarni yuklashda xatolik');
        $('#categories-table tbody').html(
          '<tr><td colspan="6" class="text-center py-3 text-danger">Server xatosi yoki API topilmadi</td></tr>'
        );
      })
      .always(function () {
        hideLoader();
      });
  }

  function safeApplyFilters() {
    try {
      applyFilters();
    } catch (e) {
      console.error('applyFilters error:', e);
      toastr.error('Front-end xatolik (filterlash)');
      $('#categories-table tbody').html(
        '<tr><td colspan="6" class="text-center py-3 text-danger">Front-end xatolik (filterlash)</td></tr>'
      );
      $('#pagination').empty();
      $('#pagination-info').text('');
    }
  }

  function applyFilters() {
    const statFilter  = $('#filter_status').val();
    const searchQuery = ($('#search').val() || '').toLowerCase();

    if (!Array.isArray(allCategories)) {
      allCategories = [];
    }

    filteredCategories = allCategories.filter(c => {
      if (statFilter !== '' && String(c.is_active) !== String(statFilter)) return false;

      if (searchQuery) {
        const txt = (c.name || '').toLowerCase();
        if (txt.indexOf(searchQuery) === -1) return false;
      }
      return true;
    });

    currentPage = 1;
    renderTablePageSafe();
  }

  function renderTablePageSafe() {
    try {
      renderTablePage();
    } catch (e) {
      console.error('renderTablePage error:', e);
      toastr.error('Front-end xatolik (jadval)');
      $('#categories-table tbody').html(
        '<tr><td colspan="6" class="text-center py-3 text-danger">Front-end xatolik (jadval)</td></tr>'
      );
      $('#pagination').empty();
      $('#pagination-info').text('');
    }
  }

  function renderTablePage() {
    const tbody = $('#categories-table tbody');
    const total = Array.isArray(filteredCategories) ? filteredCategories.length : 0;

    if (!total) {
      tbody.html('<tr><td colspan="6" class="text-center py-3 text-muted">Kategoriyalar topilmadi</td></tr>');
      $('#pagination').empty();
      $('#pagination-info').text('0 ta kategoriya');
      return;
    }

    const totalPages = Math.ceil(total / perPage);
    if (currentPage > totalPages) currentPage = totalPages;

    const start = (currentPage - 1) * perPage;
    const end   = start + perPage;
    const pageItems = filteredCategories.slice(start, end);

    let html = '';
    pageItems.forEach(c => {
      const activeBadge = c.is_active == 1
        ? '<span class="badge bg-success-subtle text-success badge-status">faol</span>'
        : '<span class="badge bg-secondary-subtle text-secondary badge-status">no-faol</span>';

      const imgHtml = c.image_path
        ? `<img src="/mark2/${escapeHtml(c.image_path)}" class="cat-img-thumb" alt="">`
        : `<div class="cat-img-thumb d-flex align-items-center justify-content-center text-muted">
            <i class="fa-regular fa-image"></i>
           </div>`;

      html += `
        <tr data-id="${c.id}">
          <td>${c.id}</td>
          <td>${escapeHtml(c.name || '')}</td>
          <td>${imgHtml}</td>
          <td>${activeBadge}</td>
          <td class="small text-muted">${escapeHtml(c.created_at || '')}</td>
          <td class="text-end">
            <div class="btn-group btn-group-sm">
              <button type="button" data-permission="categories.edit" class="btn btn-outline-primary" onclick="editCategory(${c.id})">
                <i class="fa-regular fa-pen-to-square"></i>
              </button>
              <button type="button" class="btn btn-outline-secondary" onclick="toggleCategory(${c.id})">
                ${c.is_active == 1
                  ? '<i class="fa-solid fa-eye-slash"></i>'
                  : '<i class="fa-solid fa-eye"></i>'
                }
              </button>
              <button type="button" data-permission="categories.delete" class="btn btn-outline-danger" onclick="openDeleteModal(${c.id})">
                <i class="fa-solid fa-trash-can"></i>
              </button>
            </div>
          </td>
        </tr>
      `;
    });
    tbody.html(html);

    $('#pagination-info').text(
      `${total} ta kategoriyadan ${start + 1}–${Math.min(end, total)} oralig‘i`
    );

    renderPagination(totalPages);
  }

  function renderPagination(totalPages) {
    const ul = $('#pagination');
    ul.empty();
    if (totalPages <= 1) return;

    function pageItem(page, label = null, disabled = false, active = false) {
      label = label || page;
      const liClass = 'page-item' + (disabled ? ' disabled' : '') + (active ? ' active' : '');
      const btn = `<button class="page-link" type="button" ${disabled ? 'tabindex="-1"' : ''}>${label}</button>`;
      const $li = $(`<li class="${liClass}"></li>`).append(btn);
      if (!disabled && !active) {
        $li.on('click', function () {
          currentPage = page;
          renderTablePageSafe();
        });
      }
      ul.append($li);
    }

    // Prev
    pageItem(currentPage - 1, '&laquo;', currentPage === 1, false);

    for (let i = 1; i <= totalPages; i++) {
      if (
        i === 1 ||
        i === totalPages ||
        (i >= currentPage - 1 && i <= currentPage + 1)
      ) {
        pageItem(i, i, false, i === currentPage);
      } else if (
        (i === 2 && currentPage > 3) ||
        (i === totalPages - 1 && currentPage < totalPages - 2)
      ) {
        const li = $('<li class="page-item disabled"><span class="page-link">…</span></li>');
        ul.append(li);
      }
    }

    // Next
    pageItem(currentPage + 1, '&raquo;', currentPage === totalPages, false);
  }

  // ==== Saqlash (create/update) ====
  function saveCategory() {
    const id      = $('#cat_id').val();
    const name    = $('#cat_name').val();
    const fileInp = document.getElementById('cat_image_file');
    const hasFile = fileInp && fileInp.files && fileInp.files[0];

    if (!name) {
      toastr.error('Kategoriya nomi majburiy.');
      return;
    }

    // Yangi kategoriya uchun rasm majburiy
    if (!id && !hasFile) {
      toastr.error('Yangi kategoriya uchun rasm faylini tanlang.');
      return;
    }

    const formEl   = document.getElementById('category-form');
    const formData = new FormData(formEl);
    formData.append('action', 'save');
    // is_active check
    formData.set('is_active', $('#cat_is_active').is(':checked') ? 1 : 0);

    $('#btn-save').prop('disabled', true);
    showLoader();

    $.ajax({
      url: apiUrl,
      method: 'POST',
      data: formData,
      processData: false,
      contentType: false,
      dataType: 'json'
    }).done(function (res) {
      if (!res || typeof res !== 'object') {
        toastr.error('Noto‘g‘ri server javobi');
        return;
      }
      if (res.status !== 'ok') {
        toastr.error(res.message || 'Saqlashda xatolik');
        return;
      }
      toastr.success(res.message || 'Saqlangan');
      categoryModal.hide();
      loadCategories();
    }).fail(function () {
      toastr.error('Server xatosi');
    }).always(function () {
      $('#btn-save').prop('disabled', false);
      hideLoader();
    });
  }

  function editCategory(id) {
    const c = allCategories.find(x => Number(x.id) === Number(id));
    if (!c) {
      toastr.error('Kategoriya topilmadi');
      return;
    }

    $('#cat_id').val(c.id);
    $('#cat_name').val(c.name);
    $('#cat_is_active').prop('checked', c.is_active == 1);

    $('#existing_image').val(c.image_path || '');
    $('#cat_image_file').val('');
    updateImagePreviewFromUrl(c.image_path || '');

    $('#categoryModalLabel').text('Kategoriyani tahrirlash');
    $('#form-title-text').text('Kategoriyani tahrirlash');
    $('#btn-new').show();

    categoryModal.show();
  }

  function toggleCategory(id) {
    showLoader();
    $.ajax({
      url: apiUrl,
      method: 'POST',
      dataType: 'json',
      data: { action: 'toggle', id: id }
    })
      .done(function (res) {
        if (!res || typeof res !== 'object') {
          toastr.error('Noto‘g‘ri server javobi');
          return;
        }
        if (res.status !== 'ok') {
          toastr.error(res.message || 'Xatolik');
          return;
        }
        toastr.info('Kategoriya holati yangilandi');
        loadCategories();
      })
      .fail(function () {
        toastr.error('Server xatosi (toggle)');
      })
      .always(function () {
        hideLoader();
      });
  }

  function openDeleteModal(id) {
    const c = Array.isArray(allCategories)
      ? allCategories.find(x => Number(x.id) === Number(id))
      : null;
    if (!c) {
      toastr.error('Kategoriya topilmadi');
      return;
    }
    $('#deleteCategoryId').val(c.id);
    $('#deleteCategoryName').text(c.name + ' (ID: ' + c.id + ')');
    deleteModal.show();
  }

  function confirmDelete() {
    const id = $('#deleteCategoryId').val();
    if (!id) return;

    $('#btn-confirm-delete').prop('disabled', true);
    showLoader();

    $.ajax({
      url: apiUrl,
      method: 'POST',
      dataType: 'json',
      data: { action: 'delete', id: id }
    })
      .done(function (res) {
        if (!res || typeof res !== 'object') {
          toastr.error('Noto‘g‘ri server javobi');
          return;
        }
        if (res.status !== 'ok') {
          toastr.error(res.message || 'O‘chirishda xatolik');
          return;
        }
        toastr.success('Kategoriya o‘chirildi');
        deleteModal.hide();
        loadCategories();
      })
      .fail(function () {
        toastr.error('Server xatosi (delete)');
      })
      .always(function () {
        $('#btn-confirm-delete').prop('disabled', false);
        hideLoader();
      });
  }

  function resetForm() {
    $('#cat_id').val('');
    $('#category-form')[0].reset();
    $('#cat_is_active').prop('checked', true);
    $('#existing_image').val('');
    $('#cat_image_file').val('');
    updateImagePreviewFromUrl('');

    $('#form-title-text').text('Yangi kategoriya qo‘shish');
    $('#categoryModalLabel').text('Yangi kategoriya qo‘shish');
    $('#btn-new').hide();
  }

  // ==== EXPORT ====
  function getExportData() {
    const header = [
      'ID',
      'Kategoriya',
      'Holat',
      'CreatedAt'
    ];

    const rows = (Array.isArray(filteredCategories) ? filteredCategories : []).map(c => ([
      c.id,
      c.name || '',
      c.is_active == 1 ? 'faol' : 'no-faol',
      c.created_at || ''
    ]));

    return { header, rows };
  }

  function exportData(format) {
    if (!filteredCategories || !filteredCategories.length) {
      toastr.warning('Export qilish uchun kategoriya topilmadi');
      return;
    }
    const now = new Date();
    const ts = now.toISOString().slice(0,19).replace(/[:T]/g, '-');
    const fileBase = `categories_export_${ts}`;

    switch (format) {
      case 'csv':
        exportCsv(fileBase + '.csv');
        break;
      case 'xls':
        exportExcelLike(fileBase + '.xls', 'application/vnd.ms-excel');
        break;
      case 'xlsx':
        exportExcelLike(fileBase + '.xlsx',
          'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        break;
      case 'pdf':
        exportPdf(fileBase + '.pdf');
        break;
      case 'doc':
        exportDoc(fileBase + '.doc');
        break;
      case 'png':
        exportImage(fileBase + '.png', 'image/png');
        break;
      case 'jpg':
      case 'jpeg':
        exportImage(fileBase + '.jpg', 'image/jpeg');
        break;
    }
  }

  function exportCsv(filename) {
    const { header, rows } = getExportData();
    let csv = '';
    csv += header.join(';') + '\n';
    rows.forEach(r => {
      const line = r.map(v => {
        const s = String(v ?? '');
        if (s.includes(';') || s.includes('"') || s.includes('\n')) {
          return '"' + s.replace(/"/g, '""') + '"';
        }
        return s;
      }).join(';');
      csv += line + '\n';
    });

    const blob = new Blob([csv], {type: 'text/csv;charset=utf-8;'});
    downloadBlob(blob, filename);
  }

  function exportExcelLike(filename, mimeType) {
    const { header, rows } = getExportData();
    let html = '<table border="1"><thead><tr>';
    header.forEach(h => {
      html += '<th>' + escapeHtml(h) + '</th>';
    });
    html += '</tr></thead><tbody>';
    rows.forEach(r => {
      html += '<tr>';
      r.forEach(v => {
        html += '<td>' + escapeHtml(String(v ?? '')) + '</td>';
      });
      html += '</tr>';
    });
    html += '</tbody></table>';

    const blob = new Blob([html], {type: mimeType});
    downloadBlob(blob, filename);
  }

  function exportDoc(filename) {
    const { header, rows } = getExportData();
    let tableHtml = '<table border="1" style="border-collapse:collapse;width:100%;">';
    tableHtml += '<thead><tr>';
    header.forEach(h => {
      tableHtml += '<th style="padding:4px;">' + escapeHtml(h) + '</th>';
    });
    tableHtml += '</tr></thead><tbody>';
    rows.forEach(r => {
      tableHtml += '<tr>';
      r.forEach(v => {
        tableHtml += '<td style="padding:4px;">' + escapeHtml(String(v ?? '')) + '</td>';
      });
      tableHtml += '</tr>';
    });
    tableHtml += '</tbody></table>';

    const html =
      '<html><head><meta charset="utf-8"></head><body>' +
      '<h3>Kategoriyalar eksporti</h3>' +
      tableHtml +
      '</body></html>';

    const blob = new Blob([html], {type: 'application/msword'});
    downloadBlob(blob, filename);
  }

  function exportPdf(filename) {
    const { header, rows } = getExportData();
    const { jsPDF } = window.jspdf;
    const doc = new jsPDF('l', 'pt', 'a4');

    doc.setFontSize(14);
    doc.text('Kategoriyalar eksporti', 40, 40);

    doc.autoTable({
      head: [header],
      body: rows,
      startY: 60,
      margin: { left: 40, right: 40 },
      styles: { fontSize: 8 }
    });

    doc.save(filename);
  }

  function exportImage(filename, mime) {
    const node = document.getElementById('export-area');
    html2canvas(node, {scale: 2}).then(canvas => {
      canvas.toBlob(function (blob) {
        if (!blob) {
          toastr.error('Rasm generatsiyada xatolik');
          return;
        }
        downloadBlob(blob, filename);
      }, mime, 0.95);
    }).catch(() => {
      toastr.error('Rasm generatsiyada xatolik');
    });
  }

  function downloadBlob(blob, filename) {
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = filename;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
  }

  function escapeHtml(str) {
    if (!str) return '';
    return String(str)
      .replace(/&/g, "&amp;")
      .replace(/</g, "&lt;")
      .replace(/>/g, "&gt;")
      .replace(/"/g, "&quot;")
      .replace(/'/g, "&#039;");
  }
</script>
</body>
</html>
