<?php
// public/users.php
require_once __DIR__ . '/auth_stub.php';

require_login();

$userName = current_user_name() ?: 'Admin';
?>
<!DOCTYPE html>
<html lang="uz">
<head>
  <meta charset="UTF-8">
  <title>Foydalanuvchilar – Markirovka tizimi</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">

  <!-- Bootstrap 5 -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <!-- FontAwesome -->
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css" rel="stylesheet">
  <!-- Toastr -->
  <link href="https://cdnjs.cloudflare.com/ajax/libs/toastr.js/latest/toastr.min.css" rel="stylesheet">

  <style>
    :root {
      --brand-main: #0d47a1;
      --brand-soft: #e5edff;
      --brand-accent: #22c55e;
      --bg-soft: #f3f4f8;
      --card-bg: rgba(255,255,255,0.96);
      --border-soft: #d1d9e6;
      --text-main: #0b1220;
      --text-muted: #6b7280;
    }

    body {
      min-height: 100vh;
      margin: 0;
      background: radial-gradient(circle at top left, #c7d2fe 0, transparent 45%),
                  radial-gradient(circle at bottom right, #bfdbfe 0, transparent 45%),
                  #eef2ff;
      color: var(--text-main);
    }

    .page-wrapper {
      max-width: 1200px;
      margin: 0 auto;
    }

    .navbar {
      background: rgba(13,71,161,0.96);
      backdrop-filter: blur(10px);
      box-shadow: 0 10px 25px rgba(15,23,42,0.35);
    }
    .navbar-brand, .navbar-nav .nav-link, .navbar-text {
      color: #fff !important;
    }
    .navbar-nav .nav-link {
      font-size: 0.92rem;
      opacity: 0.9;
    }
    .navbar-nav .nav-link.active {
      opacity: 1;
      font-weight: 600;
      border-bottom: 2px solid #fff;
    }

    .user-chip {
      display: inline-flex;
      align-items: center;
      gap: 0.5rem;
      padding: 0.15rem 0.75rem;
      background: rgba(15,23,42,0.2);
      border-radius: 999px;
      color: #e5edff;
      font-size: 0.85rem;
    }
    .user-avatar {
      width: 26px;
      height: 26px;
      border-radius: 999px;
      background: linear-gradient(135deg,#22c55e,#0ea5e9);
      display: inline-flex;
      align-items: center;
      justify-content: center;
      font-size: 0.9rem;
      color: #fff;
      font-weight: 700;
    }
    .user-role-badge {
      padding: 0.05rem 0.5rem;
      border-radius: 999px;
      background: rgba(15,23,42,0.35);
      font-size: 0.7rem;
      text-transform: uppercase;
      letter-spacing: 0.05em;
    }

    .card {
      border-radius: 18px;
      box-shadow: 0 16px 40px rgba(15,23,42,0.08);
      border: 1px solid rgba(148,163,184,0.35);
      background: var(--card-bg);
    }

    .form-label {
      font-weight: 600;
      color: var(--text-main);
    }

    .pill-heading {
      font-size: 0.8rem;
      text-transform: uppercase;
      letter-spacing: 0.1em;
      color: var(--text-muted);
    }

    .badge-status {
      font-size: 0.75rem;
      border-radius: 999px;
      padding: 0.2rem 0.55rem;
    }

    .table thead {
      background: #eff3ff;
      font-size: 0.88rem;
    }
    .table tbody tr {
      font-size: 0.86rem;
      transition: background 0.12s ease;
    }
    .table tbody tr:hover {
      background: #f9fafb;
    }

    .search-input {
      max-width: 260px;
    }

    .role-badge {
      font-size: 0.75rem;
      border-radius: 999px;
      padding: 0.2rem 0.55rem;
    }

    /* Global loader */
    #global-loader {
      position: fixed;
      inset: 0;
      background: rgba(15,23,42,0.25);
      display: none;
      align-items: center;
      justify-content: center;
      z-index: 2000;
    }
    #global-loader .spinner-border {
      width: 3rem;
      height: 3rem;
    }
    #global-loader .loader-text {
      color: #e5edff;
      margin-top: 0.75rem;
      font-size: 0.9rem;
    }

    @media (max-width: 767.98px) {
      .page-wrapper {
        padding-inline: 0.4rem;
      }
      .search-input {
        max-width: 100%;
      }
    }
  </style>
</head>
<body>

<?php
  $active = 'users';
  require __DIR__ . '/navbar.php';
?>

<!-- Global loader -->
<div id="global-loader" class="d-flex flex-column align-items-center justify-content-center">
  <div class="spinner-border text-light" role="status">
    <span class="visually-hidden">Yuklanmoqda...</span>
  </div>
  <div class="loader-text">Ma’lumotlar yuklanmoqda...</div>
</div>

<div class="page-wrapper">
  <div class="py-4">

    <!-- Title + create button -->
    <div class="d-flex flex-column flex-md-row justify-content-between align-items-md-center mb-3">
      <div>
        <h3 class="mb-1">
          Foydalanuvchilar boshqaruvi
        </h3>
        <div class="text-muted">
          Markirovka tizimida ishlaydigan operatorlar va adminlar.
        </div>
      </div>
      <div class="mt-3 mt-md-0 d-flex gap-2">
        <button id="btn-open-create" type="button" class="btn btn-primary">
          <i class="fa-solid fa-plus"></i> Yangi foydalanuvchi
        </button>
      </div>
    </div>

    <!-- List card + filters + export -->
    <div class="card" id="export-area">
      <div class="card-body">
        <div class="d-flex flex-column flex-xl-row justify-content-between align-items-xl-center gap-3 mb-3">
          <div>
            <div class="pill-heading mb-1">Filtrlar</div>
            <div class="d-flex flex-wrap gap-2 align-items-center">
              <!-- ✅ ROLLAR FILTRI – 4 ta rolga moslab tuzatildi -->
              <select id="filter_role" class="form-select form-select-sm" style="min-width:160px;">
                <option value="">Rol (hammasi)</option>
                <option value="1">Admin</option>
                <option value="2">Markirovkachi</option>
                <option value="3">Sotuvchi</option>
                <option value="4">Kuzatuvchi</option>
              </select>

              <select id="filter_status" class="form-select form-select-sm" style="min-width:160px;">
                <option value="">Holat (hammasi)</option>
                <option value="1">Faol</option>
                <option value="0">No-faol</option>
              </select>
            </div>
          </div>

          <div class="d-flex flex-column align-items-start align-items-xl-end gap-2">
            <div class="d-flex flex-wrap gap-2 align-items-center">
              <input type="text" id="search" class="form-control form-control-sm search-input"
                     placeholder="Qidirish (ism / login)...">
            </div>
            <div class="d-flex flex-wrap gap-2">
              <div class="btn-group">
                <button type="button" class="btn btn-outline-secondary btn-sm dropdown-toggle" data-bs-toggle="dropdown">
                  <i class="fa-solid fa-file-arrow-down"></i> Export
                </button>
                <ul class="dropdown-menu dropdown-menu-end">
                  <li><button class="dropdown-item" type="button" onclick="exportData('csv')">
                    <i class="fa-solid fa-file-csv me-2"></i>CSV
                  </button></li>
                  <li><button class="dropdown-item" type="button" onclick="exportData('xls')">
                    <i class="fa-regular fa-file-excel me-2"></i>XLS
                  </button></li>
                  <li><button class="dropdown-item" type="button" onclick="exportData('xlsx')">
                    <i class="fa-solid fa-file-excel me-2"></i>XLSX
                  </button></li>
                  <li><button class="dropdown-item" type="button" onclick="exportData('pdf')">
                    <i class="fa-regular fa-file-pdf me-2"></i>PDF
                  </button></li>
                  <li><button class="dropdown-item" type="button" onclick="exportData('doc')">
                    <i class="fa-regular fa-file-word me-2"></i>DOC
                  </button></li>
                  <li><hr class="dropdown-divider"></li>
                  <li><button class="dropdown-item" type="button" onclick="exportData('png')">
                    <i class="fa-regular fa-image me-2"></i>PNG
                  </button></li>
                  <li><button class="dropdown-item" type="button" onclick="exportData('jpg')">
                    <i class="fa-regular fa-image me-2"></i>JPG
                  </button></li>
                </ul>
              </div>
            </div>
          </div>
        </div>

        <!-- Table -->
        <div class="table-responsive" id="table-wrapper">
          <table class="table table-hover align-middle" id="users-table">
            <thead>
            <tr>
              <th>ID</th>
              <th>F.I.Sh</th>
              <th>Login</th>
              <th>Rol</th>
              <th>Holat</th>
              <th>Yaratilgan</th>
              <th>Oxirgi o‘zgarish</th>
              <th class="text-end">Amallar</th>
            </tr>
            </thead>
            <tbody>
              <tr><td colspan="8" class="text-center py-3 text-muted">Yuklanmoqda...</td></tr>
            </tbody>
          </table>
        </div>

        <!-- Pagination -->
        <div class="d-flex justify-content-between align-items-center mt-3 flex-wrap gap-2">
          <div class="text-muted small" id="pagination-info"></div>
          <nav>
            <ul class="pagination pagination-sm mb-0" id="pagination"></ul>
          </nav>
        </div>
      </div>
    </div>

  </div>
</div>

<!-- USER MODAL (Create/Edit) -->
<div class="modal fade" id="userModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-lg modal-dialog-centered">
    <form id="user-form" class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="userModalLabel">Yangi foydalanuvchi qo‘shish</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body">
        <input type="hidden" id="user_id" name="id" value="">

        <div class="row g-3">
          <div class="col-md-4">
            <label class="form-label" for="name">F.I.Sh *</label>
            <input type="text" class="form-control" id="name" name="name" required>
          </div>

          <div class="col-md-4">
            <label class="form-label" for="login">Login *</label>
            <input type="text" class="form-control" id="login" name="login" required>
          </div>

          <div class="col-md-4">
            <label class="form-label" for="password">Parol</label>
            <input type="password" class="form-control" id="password" name="password"
                   placeholder="Yangi foydalanuvchi uchun majburiy">
            <div class="form-text small text-muted">
              Yangi qo‘shilganda majburiy. Tahrirda bo‘sh qoldirsangiz – parol o‘zgarmaydi.
            </div>
          </div>

          <div class="col-md-4">
            <label class="form-label" for="role_id">Rol *</label>
            <!-- ✅ MODAL DAGI ROLLAR HAM 4 TA ROLGA MOS -->
            <select class="form-select" id="role_id" name="role_id" required>
              <option value="1">Admin</option>
              <option value="2" selected>Markirovkachi</option>
              <option value="3">Sotuvchi</option>
              <option value="4">Kuzatuvchi</option>
            </select>
          </div>

          <div class="col-md-4">
            <label class="form-label d-block">Holat</label>
            <div class="form-check form-switch">
              <input class="form-check-input" type="checkbox" role="switch" id="is_active" name="is_active" checked>
              <label class="form-check-label" for="is_active">Faol (tizimga kirish mumkin)</label>
            </div>
          </div>
        </div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Bekor qilish</button>
        <button type="submit" class="btn btn-primary" id="btn-save">
          <i class="fa-solid fa-floppy-disk"></i> Saqlash
        </button>
      </div>
    </form>
  </div>
</div>

<!-- DELETE CONFIRM MODAL -->
<div class="modal fade" id="deleteModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <div class="modal-content">
      <div class="modal-header bg-danger text-white">
        <h5 class="modal-title">
          <i class="fa-solid fa-triangle-exclamation"></i> Foydalanuvchini o‘chirish
        </h5>
        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body">
        <p class="mb-2">Mazkur foydalanuvchini rostdan ham o‘chirmoqchimisiz?</p>
        <p class="fw-semibold" id="deleteUserName"></p>
        <input type="hidden" id="deleteUserId" value="">
        <div class="alert alert-warning mb-0 small">
          <i class="fa-solid fa-circle-info"></i>
          Ehtiyot bo‘ling: foydalanuvchi boshqa jadvalda (loglar, buyurtmalar) ishlatilgan bo‘lishi mumkin.
        </div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Bekor qilish</button>
        <button type="button" class="btn btn-danger" id="btn-confirm-delete">
          <i class="fa-solid fa-trash-can"></i> O‘chirish
        </button>
      </div>
    </div>
  </div>
</div>

<!-- JS -->
<script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/toastr.js/latest/toastr.min.js"></script>

<!-- jsPDF + autoTable + html2canvas (PDF / PNG / JPG uchun) -->
<script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf-autotable/3.8.1/jspdf.plugin.autotable.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js"></script>

<script>
  const apiUrl = 'api/users.php';

  let allUsers = [];
  let filteredUsers = [];
  let currentPage = 1;
  const perPage = 10;

  let userModal;
  let deleteModal;

  toastr.options = {
    positionClass: "toast-top-right",
    timeOut: 3000
  };

  // Loader helperlari
  function showLoader() {
    const el = document.getElementById('global-loader');
    if (!el) return;
    el.style.setProperty('display', 'flex', 'important');
    el.style.opacity = '1';
  }

  function hideLoader() {
    const el = document.getElementById('global-loader');
    if (!el) return;
    el.style.setProperty('display', 'none', 'important');
    el.style.opacity = '0';
  }

  $(function () {
    userModal = new bootstrap.Modal(document.getElementById('userModal'));
    deleteModal = new bootstrap.Modal(document.getElementById('deleteModal'));

    loadUsers();

    $('#btn-open-create').on('click', function () {
      resetForm();
      $('#userModalLabel').text('Yangi foydalanuvchi qo‘shish');
      userModal.show();
    });

    $('#user-form').on('submit', function (e) {
      e.preventDefault();
      saveUser();
    });

    $('#btn-confirm-delete').on('click', function () {
      confirmDelete();
    });

    $('#filter_status').on('change', function () {
      safeApplyFilters();
    });

    $('#filter_role').on('change', function () {
      safeApplyFilters();
    });

    $('#search').on('input', function () {
      safeApplyFilters();
    });
  });

  function loadUsers() {
    $('#users-table tbody').html(
      '<tr><td colspan="8" class="text-center py-3 text-muted">Yuklanmoqda...</td></tr>'
    );

    showLoader();

    $.ajax({
      url: apiUrl,
      method: 'GET',
      dataType: 'json',
      data: { action: 'list' }
    })
      .done(function (res) {
        if (!res || typeof res !== 'object') {
          toastr.error('Noto‘g‘ri server javobi (JSON emas)');
          $('#users-table tbody').html(
            '<tr><td colspan="8" class="text-center py-3 text-danger">Serverdan noto‘g‘ri javob keldi</td></tr>'
          );
          return;
        }

        if (res.status !== 'ok') {
          toastr.error(res.message || 'Xatolik');
          $('#users-table tbody').html(
            '<tr><td colspan="8" class="text-center py-3 text-danger">Ma’lumot yuklab bo‘lmadi</td></tr>'
          );
          return;
        }

        if (!Array.isArray(res.users)) {
          toastr.error('Serverdan noto‘g‘ri formatdagi "users" keldi');
          $('#users-table tbody').html(
            '<tr><td colspan="8" class="text-center py-3 text-danger">Ma’lumot formati xato</td></tr>'
          );
          return;
        }

        allUsers = res.users;
        safeApplyFilters();
      })
      .fail(function () {
        toastr.error('Foydalanuvchilarni yuklashda xatolik');
        $('#users-table tbody').html(
          '<tr><td colspan="8" class="text-center py-3 text-danger">Server xatosi yoki API topilmadi</td></tr>'
        );
      })
      .always(function () {
        hideLoader();
      });
  }

  function safeApplyFilters() {
    try {
      applyFilters();
    } catch (e) {
      console.error('applyFilters error:', e);
      toastr.error('Front-end xatolik (filterlash)');
      $('#users-table tbody').html(
        '<tr><td colspan="8" class="text-center py-3 text-danger">Front-end xatolik (filterlash)</td></tr>'
      );
      $('#pagination').empty();
      $('#pagination-info').text('');
    }
  }

  function applyFilters() {
    const statFilter  = $('#filter_status').val();
    const roleFilter  = $('#filter_role').val();
    const searchQuery = ($('#search').val() || '').toLowerCase();

    if (!Array.isArray(allUsers)) {
      allUsers = [];
    }

    filteredUsers = allUsers.filter(u => {
      if (statFilter !== '' && String(u.is_active) !== String(statFilter)) return false;
      if (roleFilter !== '' && String(u.role_id) !== String(roleFilter)) return false;

      if (searchQuery) {
        const txt = (
          (u.name || '') + ' ' +
          (u.login || '')
        ).toLowerCase();
        if (txt.indexOf(searchQuery) === -1) return false;
      }

      return true;
    });

    currentPage = 1;
    renderTablePageSafe();
  }

  function renderTablePageSafe() {
    try {
      renderTablePage();
    } catch (e) {
      console.error('renderTablePage error:', e);
      toastr.error('Front-end xatolik (jadval)');
      $('#users-table tbody').html(
        '<tr><td colspan="8" class="text-center py-3 text-danger">Front-end xatolik (jadval)</td></tr>'
      );
      $('#pagination').empty();
      $('#pagination-info').text('');
    }
  }

  function renderTablePage() {
    const tbody = $('#users-table tbody');
    const total = Array.isArray(filteredUsers) ? filteredUsers.length : 0;

    if (!total) {
      tbody.html('<tr><td colspan="8" class="text-center py-3 text-muted">Foydalanuvchilar topilmadi</td></tr>');
      $('#pagination').empty();
      $('#pagination-info').text('0 ta foydalanuvchi');
      return;
    }

    const totalPages = Math.ceil(total / perPage);
    if (currentPage > totalPages) currentPage = totalPages;

    const start = (currentPage - 1) * perPage;
    const end   = start + perPage;
    const pageItems = filteredUsers.slice(start, end);

    let html = '';
    pageItems.forEach(u => {
      const activeBadge = u.is_active == 1
        ? '<span class="badge bg-success-subtle text-success badge-status">faol</span>'
        : '<span class="badge bg-secondary-subtle text-secondary badge-status">no-faol</span>';

      const roleHtml = roleBadge(u.role_id);

      html += `
        <tr data-id="${u.id}">
          <td>${u.id}</td>
          <td>${escapeHtml(u.name || '')}</td>
          <td><code>${escapeHtml(u.login || '')}</code></td>
          <td>${roleHtml}</td>
          <td>${activeBadge}</td>
          <td class="small text-muted">${escapeHtml(u.created_at || '')}</td>
          <td class="small text-muted">${escapeHtml(u.updated_at || '')}</td>
          <td class="text-end">
            <div class="btn-group btn-group-sm">
              <button type="button" class="btn btn-outline-primary" onclick="editUser(${u.id})">
                <i class="fa-regular fa-pen-to-square"></i>
              </button>
              <button type="button" class="btn btn-outline-secondary" onclick="toggleUser(${u.id})">
                ${u.is_active == 1
                  ? '<i class="fa-solid fa-user-slash"></i>'
                  : '<i class="fa-solid fa-user-check"></i>'
                }
              </button>
              <button type="button" class="btn btn-outline-danger" onclick="openDeleteModal(${u.id})">
                <i class="fa-solid fa-trash-can"></i>
              </button>
            </div>
          </td>
        </tr>
      `;
    });
    tbody.html(html);

    $('#pagination-info').text(
      `${total} ta foydalanuvchidan ${start + 1}–${Math.min(end, total)} oralig‘i`
    );

    renderPagination(totalPages);
  }

  function renderPagination(totalPages) {
    const ul = $('#pagination');
    ul.empty();
    if (totalPages <= 1) return;

    function pageItem(page, label = null, disabled = false, active = false) {
      label = label || page;
      const liClass = 'page-item' + (disabled ? ' disabled' : '') + (active ? ' active' : '');
      const btn = `<button class="page-link" type="button" ${disabled ? 'tabindex="-1"' : ''}>${label}</button>`;
      const $li = $(`<li class="${liClass}"></li>`).append(btn);
      if (!disabled && !active) {
        $li.on('click', function () {
          currentPage = page;
          renderTablePageSafe();
        });
      }
      ul.append($li);
    }

    // Prev
    pageItem(currentPage - 1, '&laquo;', currentPage === 1, false);

    for (let i = 1; i <= totalPages; i++) {
      if (
        i === 1 ||
        i === totalPages ||
        (i >= currentPage - 1 && i <= currentPage + 1)
      ) {
        pageItem(i, i, false, i === currentPage);
      } else if (
        (i === 2 && currentPage > 3) ||
        (i === totalPages - 1 && currentPage < totalPages - 2)
      ) {
        const li = $('<li class="page-item disabled"><span class="page-link">…</span></li>');
        ul.append(li);
      }
    }

    // Next
    pageItem(currentPage + 1, '&raquo;', currentPage === totalPages, false);
  }

  function saveUser() {
    const id = $('#user_id').val();

    const formData = {
      action: 'save',
      id: id,
      name: $('#name').val(),
      login: $('#login').val(),
      password: $('#password').val(),
      role_id: $('#role_id').val(),
      is_active: $('#is_active').is(':checked') ? 1 : 0
    };

    if (!formData.name || !formData.login) {
      toastr.error('F.I.Sh va login majburiy.');
      return;
    }
    if (!formData.id && !formData.password) {
      toastr.error('Yangi foydalanuvchi uchun parol majburiy.');
      return;
    }

    $('#btn-save').prop('disabled', true);
    showLoader();

    $.ajax({
      url: apiUrl,
      method: 'POST',
      dataType: 'json',
      data: formData
    })
      .done(function (res) {
        if (!res || res.status !== 'ok') {
          toastr.error(res && res.message ? res.message : 'Saqlashda xatolik');
          return;
        }
        toastr.success(res.message || 'Saqlangan');
        userModal.hide();
        $('#password').val('');
        loadUsers();
      })
      .fail(function () {
        toastr.error('Server xatosi (saveUser)');
      })
      .always(function () {
        $('#btn-save').prop('disabled', false);
        hideLoader();
      });
  }

  function editUser(id) {
    const u = Array.isArray(allUsers)
      ? allUsers.find(x => Number(x.id) === Number(id))
      : null;
    if (!u) {
      toastr.error('Foydalanuvchi topilmadi');
      return;
    }

    $('#user_id').val(u.id);
    $('#name').val(u.name);
    $('#login').val(u.login);
    $('#password').val('');
    $('#role_id').val(u.role_id);
    $('#is_active').prop('checked', u.is_active == 1);

    $('#userModalLabel').text('Foydalanuvchini tahrirlash');
    userModal.show();
  }

  function toggleUser(id) {
    if (!confirm('Foydalanuvchi holatini o‘zgartirasizmi?')) return;

    showLoader();
    $.ajax({
      url: apiUrl,
      method: 'POST',
      dataType: 'json',
      data: { action: 'toggle', id: id }
    })
      .done(function (res) {
        if (!res || res.status !== 'ok') {
          toastr.error(res && res.message ? res.message : 'Xatolik');
          return;
        }
        toastr.info('Foydalanuvchi holati yangilandi');
        loadUsers();
      })
      .fail(function () {
        toastr.error('Server xatosi (toggleUser)');
      })
      .always(function () {
        hideLoader();
      });
  }

  function openDeleteModal(id) {
    const u = Array.isArray(allUsers)
      ? allUsers.find(x => Number(x.id) === Number(id))
      : null;
    if (!u) {
      toastr.error('Foydalanuvchi topilmadi');
      return;
    }
    $('#deleteUserId').val(u.id);
    $('#deleteUserName').text((u.name || '') + ' (login: ' + (u.login || '') + ', ID: ' + u.id + ')');
    deleteModal.show();
  }

  function confirmDelete() {
    const id = $('#deleteUserId').val();
    if (!id) return;

    $('#btn-confirm-delete').prop('disabled', true);
    showLoader();

    $.ajax({
      url: apiUrl,
      method: 'POST',
      dataType: 'json',
      data: { action: 'delete', id: id }
    })
      .done(function (res) {
        if (!res || res.status !== 'ok') {
          toastr.error(res && res.message ? res.message : 'O‘chirishda xatolik');
          return;
        }
        toastr.success('Foydalanuvchi o‘chirildi');
        deleteModal.hide();
        loadUsers();
      })
      .fail(function () {
        toastr.error('Server xatosi (delete)');
      })
      .always(function () {
        $('#btn-confirm-delete').prop('disabled', false);
        hideLoader();
      });
  }

  function resetForm() {
    $('#user_id').val('');
    $('#user-form')[0].reset();
    $('#is_active').prop('checked', true);
    $('#role_id').val('2'); // default: Markirovkachi
    $('#password').val('');
  }

  // ---- EXPORT FUNKSIYALAR ----

  // ✅ Rol nomini 4 ta rolga moslab tuzatdik
  function getRoleName(roleId) {
    switch (Number(roleId)) {
      case 1: return 'Admin';
      case 2: return 'Markirovkachi';
      case 3: return 'Sotuvchi';
      case 4: return 'Kuzatuvchi';
      default: return '';
    }
  }

  function getExportData() {
    const header = [
      'ID',
      'Name',
      'Login',
      'Role',
      'Status',
      'CreatedAt',
      'UpdatedAt'
    ];

    const rows = (Array.isArray(filteredUsers) ? filteredUsers : []).map(u => ([

      u.id,
      u.name || '',
      u.login || '',
      getRoleName(u.role_id),
      u.is_active == 1 ? 'faol' : 'no-faol',
      u.created_at || '',
      u.updated_at || ''
    ]));

    return { header, rows };
  }

  function exportData(format) {
    if (!filteredUsers || !filteredUsers.length) {
      toastr.warning('Export qilish uchun foydalanuvchi topilmadi');
      return;
    }
    const now = new Date();
    const ts = now.toISOString().slice(0,19).replace(/[:T]/g, '-');
    const fileBase = `users_export_${ts}`;

    switch (format) {
      case 'csv':
        exportCsv(fileBase + '.csv');
        break;
      case 'xls':
        exportExcelLike(fileBase + '.xls', 'application/vnd.ms-excel');
        break;
      case 'xlsx':
        exportExcelLike(fileBase + '.xlsx',
          'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        break;
      case 'pdf':
        exportPdf(fileBase + '.pdf');
        break;
      case 'doc':
        exportDoc(fileBase + '.doc');
        break;
      case 'png':
        exportImage(fileBase + '.png', 'image/png');
        break;
      case 'jpg':
      case 'jpeg':
        exportImage(fileBase + '.jpg', 'image/jpeg');
        break;
    }
  }

  function exportCsv(filename) {
    const { header, rows } = getExportData();
    let csv = '';
    csv += header.join(';') + '\n';
    rows.forEach(r => {
      const line = r.map(v => {
        const s = String(v ?? '');
        if (s.includes(';') || s.includes('"') || s.includes('\n')) {
          return '"' + s.replace(/"/g, '""') + '"';
        }
        return s;
      }).join(';');
      csv += line + '\n';
    });

    const blob = new Blob([csv], {type: 'text/csv;charset=utf-8;'});
    downloadBlob(blob, filename);
  }

  function exportExcelLike(filename, mimeType) {
    const { header, rows } = getExportData();
    let html = '<table border="1"><thead><tr>';
    header.forEach(h => {
      html += '<th>' + escapeHtml(h) + '</th>';
    });
    html += '</tr></thead><tbody>';
    rows.forEach(r => {
      html += '<tr>';
      r.forEach(v => {
        html += '<td>' + escapeHtml(String(v ?? '')) + '</td>';
      });
      html += '</tr>';
    });
    html += '</tbody></table>';

    const blob = new Blob([html], {type: mimeType});
    downloadBlob(blob, filename);
  }

  function exportDoc(filename) {
    const { header, rows } = getExportData();
    let tableHtml = '<table border="1" style="border-collapse:collapse;width:100%;">';
    tableHtml += '<thead><tr>';
    header.forEach(h => {
      tableHtml += '<th style="padding:4px;">' + escapeHtml(h) + '</th>';
    });
    tableHtml += '</tr></thead><tbody>';
    rows.forEach(r => {
      tableHtml += '<tr>';
      r.forEach(v => {
        tableHtml += '<td style="padding:4px;">' + escapeHtml(String(v ?? '')) + '</td>';
      });
      tableHtml += '</tr>';
    });
    tableHtml += '</tbody></table>';

    const html =
      '<html><head><meta charset="utf-8"></head><body>' +
      '<h3>Foydalanuvchilar eksporti</h3>' +
      tableHtml +
      '</body></html>';

    const blob = new Blob([html], {type: 'application/msword'});
    downloadBlob(blob, filename);
  }

  function exportPdf(filename) {
    const { header, rows } = getExportData();
    const { jsPDF } = window.jspdf;
    const doc = new jsPDF('l', 'pt', 'a4');

    doc.setFontSize(14);
    doc.text('Foydalanuvchilar eksporti', 40, 40);

    doc.autoTable({
      head: [header],
      body: rows,
      startY: 60,
      margin: { left: 40, right: 40 },
      styles: { fontSize: 8 }
    });

    doc.save(filename);
  }

  function exportImage(filename, mime) {
    const node = document.getElementById('export-area');
    html2canvas(node, {scale: 2}).then(canvas => {
      canvas.toBlob(function (blob) {
        if (!blob) {
          toastr.error('Rasm generatsiyada xatolik');
          return;
        }
        downloadBlob(blob, filename);
      }, mime, 0.95);
    }).catch(() => {
      toastr.error('Rasm generatsiyada xatolik');
    });
  }

  function downloadBlob(blob, filename) {
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = filename;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
  }

  // ✅ Jadvalda rol ko‘rsatish – 4 ta rolga moslandi
  function roleBadge(roleId) {
    switch (Number(roleId)) {
      case 1:
        return '<span class="badge bg-danger-subtle text-danger role-badge"><i class="fa-solid fa-crown"></i> Admin</span>';
      case 2:
        return '<span class="badge bg-primary-subtle text-primary role-badge"><i class="fa-solid fa-barcode"></i> Markirovkachi</span>';
      case 3:
        return '<span class="badge bg-warning-subtle text-warning role-badge"><i class="fa-solid fa-cart-shopping"></i> Sotuvchi</span>';
      case 4:
      default:
        return '<span class="badge bg-secondary-subtle text-secondary role-badge"><i class="fa-solid fa-eye"></i> Kuzatuvchi</span>';
    }
  }

  function escapeHtml(str) {
    if (!str) return '';
    return String(str)
      .replace(/&/g, "&amp;")
      .replace(/</g, "&lt;")
      .replace(/>/g, "&gt;")
      .replace(/"/g, "&quot;")
      .replace(/'/g, "&#039;");
  }
</script>
</body>
</html>
