(function(){
  const csrf = document.body.getAttribute('data-csrf') || '';
  const fmt = new Intl.NumberFormat('uz-UZ');
  const money = (v)=> fmt.format(Number(v || 0).toFixed(0));

  // -------------- Auth (login) --------------
  const loginForm = document.getElementById('loginForm');
  if (loginForm) {
    loginForm.addEventListener('submit', async (e)=>{
      e.preventDefault();
      const fd = new FormData(loginForm);
      const res = await fetch('/api/auth.php?action=login', {method:'POST', body: fd});
      const data = await res.json();
      if (!res.ok) {
        document.getElementById('loginError').classList.remove('d-none');
        document.getElementById('loginError').textContent = data.error || 'Kirishda xatolik';
        return;
      }
      location.href = '/dashboard.php';
    });
  }

  // -------------- Dashboard stats --------------
  if (location.pathname.endsWith('/dashboard.php')) {
    fetch('/api/dashboard.php')
      .then(r=>r.json())
      .then(d=>{
        document.getElementById('statProducts').textContent = d.products;
        document.getElementById('statCustomers').textContent = d.customers;
        document.getElementById('statInvoices').textContent = d.invoices;
        document.getElementById('statSales').textContent = money(d.total_sales);
      });
  }

  // Utility to build pagination
  function buildPagination(elem, total, page, per, onClick) {
    elem.innerHTML = '';
    const pages = Math.max(1, Math.ceil(total/per));
    const mk = (p, label, active=false, disabled=false)=>{
      const li = document.createElement('li');
      li.className = 'page-item ' + (active?'active ':'') + (disabled?'disabled':'');
      const a = document.createElement('a');
      a.className = 'page-link'; a.href='#'; a.textContent = label;
      a.addEventListener('click', (e)=>{e.preventDefault(); if(!disabled) onClick(p);});
      li.appendChild(a); elem.appendChild(li);
    };
    mk(page-1, '«', false, page===1);
    for (let i=Math.max(1,page-2); i<=Math.min(pages,page+2); i++) {
      mk(i, i, i===page, false);
    }
    mk(page+1, '»', false, page===pages);
  }

  // -------------- Products Page --------------
  if (document.querySelector('[data-page="products"]')) {
    let state = {page:1, per:10, q:''};
    const tbody = document.getElementById('prodTbody');
    const info  = document.getElementById('prodInfo');
    const pag   = document.getElementById('prodPagination');
    const modal = new bootstrap.Modal('#modalProduct');
    const form  = document.getElementById('productForm');
    const error = document.getElementById('prodError');
    document.getElementById('prodSearchBtn').addEventListener('click', ()=>{
      state.q = document.getElementById('prodSearch').value.trim(); state.page=1; load();
    });

    async function load(){
      const res = await fetch(`/api/products.php?action=list&page=${state.page}&per_page=${state.per}&q=${encodeURIComponent(state.q)}`);
      const d = await res.json();
      tbody.innerHTML = '';
      d.data.forEach(r=>{
        const tr = document.createElement('tr');
        tr.innerHTML = `
          <td>${r.id}</td>
          <td>${escapeHtml(r.name)}</td>
          <td>${escapeHtml(r.sku||'')}</td>
          <td>${money(r.price)}</td>
          <td>${r.stock}</td>
          <td class="text-nowrap">
            <button class="btn btn-sm btn-outline-primary me-1" data-act="edit" data-id="${r.id}"><i class="bi bi-pencil"></i></button>
            <button class="btn btn-sm btn-outline-danger" data-act="del" data-id="${r.id}"><i class="bi bi-trash"></i></button>
          </td>`;
        tbody.appendChild(tr);
      });
      info.textContent = `Jami: ${d.total} ta`;
      buildPagination(pag, d.total, d.page, d.per_page, (p)=>{state.page=p; load();});
    }
    load();

    tbody.addEventListener('click', async (e)=>{
      const btn = e.target.closest('button'); if(!btn) return;
      const id = btn.getAttribute('data-id');
      const act = btn.getAttribute('data-act');
      if (act==='edit') {
        const r = await (await fetch('/api/products.php?action=get&id='+id)).json();
        document.getElementById('prodId').value = r.id;
        document.getElementById('prodName').value = r.name;
        document.getElementById('prodSku').value = r.sku || '';
        document.getElementById('prodPrice').value = r.price;
        document.getElementById('prodStock').value = r.stock;
        document.getElementById('prodNotes').value = r.notes || '';
        document.getElementById('modalProductTitle').textContent = 'Mahsulotni tahrirlash';
        error.classList.add('d-none');
        modal.show();
      } else if (act==='del') {
        if (!confirm('O‘chirishni tasdiqlaysizmi?')) return;
        const fd = new FormData();
        fd.append('id', id);
        await fetch('/api/products.php?action=delete', {method:'POST', body: fd, headers: {'X-CSRF-TOKEN': getCsrf()}}).then(r=>r.json());
        load();
      }
    });

    form.addEventListener('submit', async (e)=>{
      e.preventDefault();
      error.classList.add('d-none');
      const fd = new FormData(form);
      const action = fd.get('id') ? 'update' : 'create';
      const res = await fetch('/api/products.php?action='+action, {method:'POST', body: fd, headers: {'X-CSRF-TOKEN': getCsrf()}});
      const d = await res.json();
      if (!res.ok) {
        error.textContent = d.error || 'Xatolik'; error.classList.remove('d-none'); return;
      }
      form.reset(); modal.hide(); load();
    });

    // Open modal for create
    document.querySelector('[data-bs-target="#modalProduct"]').addEventListener('click', ()=>{
      document.getElementById('modalProductTitle').textContent = 'Yangi mahsulot';
      form.reset(); document.getElementById('prodId').value = ''; error.classList.add('d-none');
    });
  }

  // -------------- Customers Page --------------
  if (document.querySelector('[data-page="customers"]')) {
    let state = {page:1, per:10, q:''};
    const tbody = document.getElementById('custTbody');
    const info  = document.getElementById('custInfo');
    const pag   = document.getElementById('custPagination');
    const modal = new bootstrap.Modal('#modalCustomer');
    const form  = document.getElementById('customerForm');
    const error = document.getElementById('custError');
    document.getElementById('custSearchBtn').addEventListener('click', ()=>{
      state.q = document.getElementById('custSearch').value.trim(); state.page=1; load();
    });

    async function load(){
      const res = await fetch(`/api/customers.php?action=list&page=${state.page}&per_page=${state.per}&q=${encodeURIComponent(state.q)}`);
      const d = await res.json();
      tbody.innerHTML = '';
      d.data.forEach(r=>{
        const tr = document.createElement('tr');
        tr.innerHTML = `
          <td>${r.id}</td>
          <td>${escapeHtml(r.name)}</td>
          <td>${escapeHtml(r.phone||'')}</td>
          <td>${escapeHtml(r.email||'')}</td>
          <td>${escapeHtml(r.address||'')}</td>
          <td class="text-nowrap">
            <button class="btn btn-sm btn-outline-primary me-1" data-act="edit" data-id="${r.id}"><i class="bi bi-pencil"></i></button>
            <button class="btn btn-sm btn-outline-danger" data-act="del" data-id="${r.id}"><i class="bi bi-trash"></i></button>
          </td>`;
        tbody.appendChild(tr);
      });
      info.textContent = `Jami: ${d.total} ta`;
      buildPagination(pag, d.total, d.page, d.per_page, (p)=>{state.page=p; load();});
    }
    load();

    tbody.addEventListener('click', async (e)=>{
      const btn = e.target.closest('button'); if(!btn) return;
      const id = btn.getAttribute('data-id');
      const act = btn.getAttribute('data-act');
      if (act==='edit') {
        const r = await (await fetch('/api/customers.php?action=get&id='+id)).json();
        document.getElementById('custId').value = r.id;
        document.getElementById('custName').value = r.name;
        document.getElementById('custPhone').value = r.phone || '';
        document.getElementById('custEmail').value = r.email || '';
        document.getElementById('custAddress').value = r.address || '';
        document.getElementById('modalCustomerTitle').textContent = 'Mijozni tahrirlash';
        error.classList.add('d-none');
        modal.show();
      } else if (act==='del') {
        if (!confirm('O‘chirishni tasdiqlaysizmi?')) return;
        const fd = new FormData();
        fd.append('id', id);
        await fetch('/api/customers.php?action=delete', {method:'POST', body: fd, headers: {'X-CSRF-TOKEN': getCsrf()}}).then(r=>r.json());
        load();
      }
    });

    form.addEventListener('submit', async (e)=>{
      e.preventDefault();
      error.classList.add('d-none');
      const fd = new FormData(form);
      const action = fd.get('id') ? 'update' : 'create';
      const res = await fetch('/api/customers.php?action='+action, {method:'POST', body: fd, headers: {'X-CSRF-TOKEN': getCsrf()}});
      const d = await res.json();
      if (!res.ok) {
        error.textContent = d.error || 'Xatolik'; error.classList.remove('d-none'); return;
      }
      form.reset(); modal.hide(); load();
    });

    document.querySelector('[data-bs-target="#modalCustomer"]').addEventListener('click', ()=>{
      document.getElementById('modalCustomerTitle').textContent = 'Yangi mijoz';
      form.reset(); document.getElementById('custId').value = ''; error.classList.add('d-none');
    });
  }

  // -------------- Invoices Page --------------
  if (document.querySelector('[data-page="invoices"]')) {
    let state = {page:1, per:10, q:''};
    const tbody = document.getElementById('invTbody');
    const info  = document.getElementById('invInfo');
    const pag   = document.getElementById('invPagination');
    const modal = new bootstrap.Modal('#modalInvoice');
    const modalView = new bootstrap.Modal('#modalInvoiceView');
    const errEl = document.getElementById('invError');

    document.getElementById('invSearchBtn').addEventListener('click', ()=>{
      state.q = document.getElementById('invSearch').value.trim(); state.page=1; load();
    });

    async function load(){
      const res = await fetch(`/api/invoices.php?action=list&page=${state.page}&per_page=${state.per}&q=${encodeURIComponent(state.q)}`);
      const d = await res.json();
      tbody.innerHTML = '';
      d.data.forEach(r=>{
        const tr = document.createElement('tr');
        tr.innerHTML = `
          <td>${r.id}</td>
          <td>${escapeHtml(r.customer_name)}</td>
          <td>${r.invoice_date}</td>
          <td>${money(r.total_amount)}</td>
          <td class="text-nowrap">
            <button class="btn btn-sm btn-outline-secondary me-1" data-act="view" data-id="${r.id}"><i class="bi bi-eye"></i></button>
            <button class="btn btn-sm btn-outline-primary me-1" data-act="edit" data-id="${r.id}"><i class="bi bi-pencil"></i></button>
            <button class="btn btn-sm btn-outline-danger" data-act="del" data-id="${r.id}"><i class="bi bi-trash"></i></button>
          </td>`;
        tbody.appendChild(tr);
      });
      info.textContent = `Jami: ${d.total} ta`;
      buildPagination(pag, d.total, d.page, d.per_page, (p)=>{state.page=p; load();});
    }
    load();

    // Prepare selects & items
    let productsCache = [];
    let customersCache = [];

    async function loadProducts(){
      const r = await fetch('/api/products.php?action=list&per_page=1000');
      const d = await r.json();
      productsCache = d.data || [];
    }
    async function loadCustomers(){
      const r = await fetch('/api/customers.php?action=list&per_page=1000');
      const d = await r.json();
      customersCache = d.data || [];
    }

    function fillCustomerSelect(sel, value=null){
      sel.innerHTML = '<option value="">-- tanlang --</option>';
      customersCache.forEach(c=>{
        const opt = document.createElement('option');
        opt.value = c.id; opt.textContent = c.name;
        if (value && Number(value)===Number(c.id)) opt.selected = true;
        sel.appendChild(opt);
      });
    }

    function addItemRow(container, row={product_id:'', quantity:1, unit_price:0}){
      const tr = document.createElement('tr');
      tr.innerHTML = `
        <td>
          <select class="form-select form-select-sm productSel">
            <option value="">-- tanlang --</option>
          </select>
        </td>
        <td><input type="number" class="form-control form-control-sm qty" value="${row.quantity}" min="1"></td>
        <td><input type="number" class="form-control form-control-sm price" value="${row.unit_price}" step="0.01"></td>
        <td class="line-total">0</td>
        <td><button class="btn btn-sm btn-outline-danger delRow"><i class="bi bi-x-lg"></i></button></td>
      `;
      container.appendChild(tr);
      const sel = tr.querySelector('.productSel');
      productsCache.forEach(p=>{
        const opt = document.createElement('option');
        opt.value = p.id; opt.textContent = `${p.name} (${p.sku||''})`;
        if (row.product_id && Number(row.product_id)===Number(p.id)) opt.selected = true;
        sel.appendChild(opt);
      });
      const qty = tr.querySelector('.qty');
      const price = tr.querySelector('.price');
      const line = tr.querySelector('.line-total');
      function recalc(){
        const q = Number(qty.value||0), pr = Number(price.value||0);
        line.textContent = money(q*pr);
        recalcTotal();
      }
      sel.addEventListener('change', ()=>{
        const p = productsCache.find(pp=>Number(pp.id)===Number(sel.value));
        if (p) { price.value = p.price; recalc(); }
      });
      qty.addEventListener('input', recalc);
      price.addEventListener('input', recalc);
      tr.querySelector('.delRow').addEventListener('click', (e)=>{e.preventDefault(); tr.remove(); recalcTotal();});
      recalc();
    }

    function recalcTotal(){
      let sum = 0;
      document.querySelectorAll('#invoiceItems tr').forEach(tr=>{
        const line = tr.querySelector('.line-total').textContent.replace(/\s/g,'');
        const val = Number(line.split(',').join(''));
        // Fallback if parsing fails
        const qty = Number(tr.querySelector('.qty').value||0);
        const price = Number(tr.querySelector('.price').value||0);
        sum += (val || (qty*price));
      });
      document.getElementById('invoiceTotal').textContent = money(sum);
    }

    function serializeItems(){
      const items = [];
      document.querySelectorAll('#invoiceItems tr').forEach(tr=>{
        const product_id = Number(tr.querySelector('.productSel').value);
        const quantity = Number(tr.querySelector('.qty').value||0);
        const unit_price = Number(tr.querySelector('.price').value||0);
        if (product_id && quantity>0) items.push({product_id, quantity, unit_price});
      });
      return items;
    }

    // Add button
    document.getElementById('addItemBtn').addEventListener('click', (e)=>{
      e.preventDefault();
      addItemRow(document.getElementById('invoiceItems'));
    });

    // Open create modal
    document.querySelector('[data-bs-target="#modalInvoice"]').addEventListener('click', async ()=>{
      errEl.classList.add('d-none');
      if (productsCache.length===0) await loadProducts();
      if (customersCache.length===0) await loadCustomers();
      document.getElementById('modalInvoiceTitle').textContent = 'Yangi hisob-faktura';
      document.getElementById('invoiceId').value = '';
      document.getElementById('invoiceDate').valueAsDate = new Date();
      document.getElementById('invoiceNotes').value = '';
      const sel = document.getElementById('invoiceCustomer');
      fillCustomerSelect(sel);
      const items = document.getElementById('invoiceItems');
      items.innerHTML = '';
      addItemRow(items);
    });

    // Save invoice
    document.getElementById('saveInvoiceBtn').addEventListener('click', async ()=>{
      const payload = {
        id: document.getElementById('invoiceId').value || undefined,
        customer_id: Number(document.getElementById('invoiceCustomer').value||0),
        invoice_date: document.getElementById('invoiceDate').value,
        notes: document.getElementById('invoiceNotes').value || '',
        items: serializeItems()
      };
      const action = payload.id ? 'update' : 'create';
      const res = await fetch('/api/invoices.php?action='+action, {
        method: 'POST',
        headers: {'Content-Type':'application/json','X-CSRF-TOKEN': getCsrf()},
        body: JSON.stringify(payload)
      });
      const d = await res.json();
      if (!res.ok) {
        errEl.textContent = d.error || 'Xatolik'; errEl.classList.remove('d-none'); return;
      }
      modal.hide(); load();
    });

    // Row actions (view / edit / delete)
    tbody.addEventListener('click', async (e)=>{
      const btn = e.target.closest('button'); if(!btn) return;
      const id = btn.getAttribute('data-id');
      const act = btn.getAttribute('data-act');
      if (act==='view') {
        const r = await (await fetch('/api/invoices.php?action=get&id='+id)).json();
        showInvoiceView(r);
      } else if (act==='edit') {
        errEl.classList.add('d-none');
        const r = await (await fetch('/api/invoices.php?action=get&id='+id)).json();
        if (productsCache.length===0) await loadProducts();
        if (customersCache.length===0) await loadCustomers();
        document.getElementById('modalInvoiceTitle').textContent = `Tahrirlash #${r.id}`;
        document.getElementById('invoiceId').value = r.id;
        document.getElementById('invoiceDate').value = r.invoice_date;
        document.getElementById('invoiceNotes').value = r.notes || '';
        fillCustomerSelect(document.getElementById('invoiceCustomer'), r.customer_id);
        const items = document.getElementById('invoiceItems');
        items.innerHTML = '';
        (r.items||[]).forEach(it=> addItemRow(items, {product_id: it.product_id, quantity: it.quantity, unit_price: it.unit_price}) );
        modal.show();
      } else if (act==='del') {
        if (!confirm('O‘chirishni tasdiqlaysizmi?')) return;
        const fd = new FormData();
        fd.append('id', id);
        await fetch('/api/invoices.php?action=delete', {method:'POST', body: fd, headers: {'X-CSRF-TOKEN': getCsrf()}}).then(r=>r.json());
        load();
      }
    });

    function showInvoiceView(r){
      const el = document.getElementById('invoiceViewBody');
      const rows = (r.items||[]).map(it=>`
        <tr>
          <td>${escapeHtml(it.product_name)} <span class="text-secondary small">${escapeHtml(it.sku||'')}</span></td>
          <td>${it.quantity}</td>
          <td>${money(it.unit_price)}</td>
          <td>${money(it.line_total)}</td>
        </tr>
      `).join('');
      el.innerHTML = `
        <div class="mb-2"><strong>Mijoz:</strong> ${escapeHtml(r.customer_name)}</div>
        <div class="mb-2"><strong>Sana:</strong> ${r.invoice_date}</div>
        <div class="table-responsive">
          <table class="table table-sm">
            <thead><tr><th>Mahsulot</th><th>Miqdor</th><th>Narx</th><th>Jami</th></tr></thead>
            <tbody>${rows}</tbody>
          </table>
        </div>
        <div class="text-end"><strong>Jami:</strong> ${money(r.total_amount)}</div>
      `;
      modalView.show();
    }
  }

  function escapeHtml(s){ return (s||'').replace(/[&<>"']/g, (m)=>({ '&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;' }[m])); }
  function getCsrf(){ return document.body.getAttribute('data-csrf') || ''; }
})();
